/*:
 * @target MZ
 * @plugindesc Backlog Core（表示専用）v1.3.1 — 3行/フォント調整/制御文字OK/右クリック安全。スイッチで開閉可。表示中はイベント停止可。
 * @author Human
 * @help
 * ■使い方（新機能）
 * - パラメータ「バックログを開くスイッチ」「閉じるスイッチ」にIDを設定。
 * - そのスイッチをONにすると、その瞬間に開く/閉じる。既定で自動OFF（変更可）。
 * - 既存の「有効スイッチ」が設定されている場合、ONのときだけ開けます。
 *
 * ■推奨並び順
 * SimpleVoice.js → HS_VoiceLinker_SimpleVoice.js → HS_BacklogCore_Simple.js
 *
 * @param maxEntries
 * @text 最大記録数
 * @type number
 * @min 50 @max 2000
 * @default 300
 *
 * @param showNameFirst
 * @text 名前を表示する
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param separateNameLine
 * @text 名前を別行に表示
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param nameFontSize
 * @text 名前フォントサイズ
 * @type number
 * @min 10 @max 64
 * @default 22
 *
 * @param textFontSize
 * @text 本文フォントサイズ(基準)
 * @type number
 * @min 10 @max 64
 * @default 24
 *
 * @param entryLines
 * @text 本文の最大行数
 * @type number
 * @min 1 @max 5
 * @default 3
 *
 * @param shrinkToFit
 * @text はみ出すと自動縮小
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param entryPaddingY
 * @text 行ボックス上下余白(px)
 * @type number
 * @min 0 @max 32
 * @default 8
 *
 * @param entryTextOffsetY
 * @text 本文Yオフセット(±px)
 * @type number
 * @min -64 @max 64
 * @default 0
 *
 * @param bulletRightPadding
 * @text 右側の●マーク余白(px)
 * @type number
 * @min 0 @max 64
 * @default 28
 *
 * @param overlayOffsetY
 * @text ウィンドウ全体のYオフセット(±px)
 * @type number
 * @min -200 @max 200
 * @default 0
 *
 * @param dimOpacity
 * @text 暗転の濃さ(0-255)
 * @type number
 * @min 0 @max 255
 * @default 96
 *
 * @param hotkey
 * @text ホットキー（none=無効）
 * @type select
 * @option none
 * @option pageup
 * @option pagedown
 * @option shift
 * @option control
 * @option tab
 * @option cancel
 * @option menu
 * @default none
 *
 * @param enableSwitchId
 * @text 有効スイッチ（0=無視）
 * @type switch
 * @default 0
 *
 * @param lockInterpreters
 * @text 表示中：イベント処理停止（並列/コモン/実行中）
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param lockMapEvents
 * @text 表示中：マップイベント更新停止（起動抑止）
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param lockPlayer
 * @text 表示中：プレイヤー更新停止（移動/接触抑止）
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param openWithSwitchId
 * @text バックログを開くスイッチ（0=無効）
 * @type switch
 * @default 0
 *
 * @param openSwitchAutoReset
 * @text 開いたら自動でOFFにする
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param closeWithSwitchId
 * @text バックログを閉じるスイッチ（0=無効）
 * @type switch
 * @default 0
 *
 * @param closeSwitchAutoReset
 * @text 閉じたら自動でOFFにする
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param blockWhenOptionsOpen
 * @text オプション表示中は開けない
 * @type boolean
 * @on はい @off いいえ
 * @default true
 *
 * @param optionsOpenSwitchId
 * @text オプション表示中スイッチ（0=自動判定）
 * @type switch
 * @default 0
 *
 * @command OpenBacklog
 * @text バックログを開く
 *
 * @command CloseBacklog
 * @text バックログを閉じる
 *
 * @command ClearBacklog
 * @text バックログ消去
 */
(() => {
  'use strict';

  // ===== Params =====
  const PN = (document.currentScript && document.currentScript.src ? (document.currentScript.src.match(/([^\/]+)\.js$/i) || [])[1] : null) || 'HS_BacklogCore_Simple';
  const P  = PluginManager.parameters(PN);

  const MAX_ENTRIES   = +P.maxEntries || 300;
  const SHOW_NAME     = P.showNameFirst === 'true';
  const SEPARATE_NAME = P.separateNameLine === 'true';
  const NAME_FS       = +P.nameFontSize || 22;
  const TEXT_FS_BASE  = +P.textFontSize || 24;
  const ENTRY_LINES   = Math.max(1, +P.entryLines || 3);
  const SHRINK        = P.shrinkToFit !== 'false';
  const PADY          = +P.entryPaddingY || 8;
  const TEXT_OFSY     = +P.entryTextOffsetY || 0;
  const BULLET_RPAD   = +P.bulletRightPadding || 28;
  const DIM_OPACITY   = +P.dimOpacity || 96;
  const HOTKEY        = String(P.hotkey || 'none');

  const ENABLE_SW     = +P.enableSwitchId || 0;
  const LOCK_INT      = P.lockInterpreters !== 'false';
  const LOCK_EVENTS   = P.lockMapEvents !== 'false';
  const LOCK_PLAYER   = P.lockPlayer !== 'false';

  const OPEN_SW       = +P.openWithSwitchId || 0;
  const OPEN_AUTO_OFF = P.openSwitchAutoReset !== 'false';
  const CLOSE_SW      = +P.closeWithSwitchId || 0;
  const CLOSE_AUTO_OFF= P.closeSwitchAutoReset !== 'false';

  const BLOCK_WHEN_OPTIONS = P.blockWhenOptionsOpen !== 'false';
  const OPTIONS_OPEN_SW    = +P.optionsOpenSwitchId || 0;

  const OVERLAY_OY    = +P.overlayOffsetY || 0;

  const allowOpen = () => ENABLE_SW === 0 || $gameSwitches.value(ENABLE_SW);


  // Hotkey is disabled on non-game scenes (e.g., Title) to avoid opening backlog there.
  const isHotkeySceneAllowed = () => {
    const s = SceneManager._scene;
    if (!s) return false;
    try {
      if (window.Scene_Boot && s instanceof Scene_Boot) return false;
      if (window.Scene_Title && s instanceof Scene_Title) return false;
      if (window.Scene_Gameover && s instanceof Scene_Gameover) return false;
      if (window.Scene_Options && s instanceof Scene_Options) return false;
      if (window.Scene_Save && s instanceof Scene_Save) return false;
      if (window.Scene_Load && s instanceof Scene_Load) return false;
    } catch (_) {}
    return true;
  };

  // Options overlay guard (prevents backlog from opening over options UI).
  const isOptionsOpenNow = () => {
    if (!BLOCK_WHEN_OPTIONS) return false;
    try {
      if (OPTIONS_OPEN_SW > 0 && $gameSwitches && $gameSwitches.value(OPTIONS_OPEN_SW)) return true;
      if (window.HS && typeof window.HS.isOptionsOverlayOpen === 'function' && window.HS.isOptionsOverlayOpen()) return true;
      const s = SceneManager._scene;
      if (!s) return false;
      try { if (window.Scene_Options && s instanceof Scene_Options) return true; } catch(_) {}
      // Detect standard Window_Options overlay in current scene (rare, but safe).
      if (window.Window_Options && s._windowLayer && s._windowLayer.children){
        for (const ch of s._windowLayer.children){
          try { if (ch && ch.visible && ch instanceof Window_Options) return true; } catch(_) {}
        }
      }
    } catch(_) {}
    return false;
  };



  // ===== shared namespace =====
  window.HSBL = window.HSBL || {};
  const HSBL = window.HSBL;
  HSBL._openFlag = HSBL._openFlag || false;
  HSBL.isOpen = () => !!HSBL._openFlag;

  // block context menu
  if (typeof document !== 'undefined' && !document._hsbl_ctxmenu_blocked){
    document.addEventListener('contextmenu', e => e.preventDefault());
    document._hsbl_ctxmenu_blocked = true;
  }

  // ===== save-time light conversion =====
  function conv(text){
    if (!text) return '';
    try{
      text = text.replace(/\\V\[(\d+)\]/gi, (_,n)=> String($gameVariables.value(+n) ?? 0));
      text = text.replace(/\\N\[(\d+)\]/gi, (_,n)=> { const a=$gameActors.actor(+n); return a? a.name() : ''; });
      text = text.replace(/\\P\[(\d+)\]/gi, (_,n)=> { const m=$gameParty.members()[+n-1]; return m? m.name() : ''; });
      return text
        .replace(/\\C\[\d+\]/gi,'').replace(/\\I\[\d+\]/gi,'').replace(/\\FS\[\d+\]/gi,'')
        .replace(/\\\\/g,'\\')
        .replace(/\\\{|\\\}|\\\.|\\\||\\\!|\\\>|\\\<|\\\^|\\\#|\\G|\\W\[\d+\]/gi,'');
    }catch(_){ return String(text); }
  }

  // ===== storage =====
  const _Sys_init = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function(){
    _Sys_init.apply(this, arguments);
    this._hsBacklog = [];
  };
  Game_System.prototype.hsBacklog = function(){ return this._hsBacklog; };
  Game_System.prototype.hsPushBacklog = function(e){ const a=this._hsBacklog; a.push(e); if (a.length > MAX_ENTRIES) a.shift(); };
  Game_System.prototype.hsClearBacklog = function(){ this._hsBacklog = []; };

  // ===== capture $gameMessage =====
  const Buf = {
    active:false, lines:[], speaker:'', faceName:'', faceIndex:0, voice:null,
    reset(){ this.active=false; this.lines.length=0; this.speaker=''; this.faceName=''; this.faceIndex=0; this.voice=null; }
  };

  const _GM_add = Game_Message.prototype.add;
  Game_Message.prototype.add = function(text){
    _GM_add.apply(this, arguments);
    try{
      if (!Buf.active){
        Buf.active   = true;
        Buf.speaker  = this._speakerName || '';
        Buf.faceName = this._faceName || '';
        Buf.faceIndex= +(this._faceIndex || 0);
        if (window.HSBL && typeof window.HSBL.consumeReservation === 'function'){
          Buf.voice = window.HSBL.consumeReservation();
        }
      }
      Buf.lines.push(String(text));
    }catch(_){}
  };

  const _GM_clear = Game_Message.prototype.clear;
  Game_Message.prototype.clear = function(){
    try{
      if (Buf.active){
        const text = conv(Buf.lines.join('\n')).trim();
        if (text){
          $gameSystem.hsPushBacklog({
            speaker:  Buf.speaker || '',
            text:     text,
            faceName: Buf.faceName || '',
            faceIndex: Number.isFinite(Buf.faceIndex)? Buf.faceIndex : 0,
            voice: Buf.voice ? {
              name:String(Buf.voice.name||''), volume:+(Buf.voice.volume??100),
              pitch:+(Buf.voice.pitch??100), pan:+(Buf.voice.pan??0),
              loop:!!Buf.voice.loop, channel:+(Buf.voice.channel??0)
            } : null
          });
        }
      }
    }catch(e){ console.warn('[HSBL-Core] commit error:', e); }
    Buf.reset();
    _GM_clear.apply(this, arguments);
  };

  // ===== Window =====
  function Window_HS_BacklogList(){ this.initialize.apply(this, arguments); }
  Window_HS_BacklogList.prototype = Object.create(Window_Selectable.prototype);
  Window_HS_BacklogList.prototype.constructor = Window_HS_BacklogList;

  Window_HS_BacklogList.prototype.standardFontSize = function(){ return TEXT_FS_BASE; };
  Window_HS_BacklogList.prototype.lineHeight = function(){ return Math.floor((this.contents?.fontSize || TEXT_FS_BASE) + 8); };
  Window_HS_BacklogList.prototype.itemHeight = function(){
    const nameH = (SHOW_NAME && SEPARATE_NAME) ? (NAME_FS + 6) : 0;
    const lineH = (this.contents?.fontSize || TEXT_FS_BASE) + 8;
    return Math.round(PADY*2 + nameH + lineH * ENTRY_LINES);
  };

  Window_HS_BacklogList.prototype.initialize = function(rect){
    Window_Selectable.prototype.initialize.call(this, rect);
    this._data = $gameSystem.hsBacklog();
    this.setHandler('ok',     this.onOk.bind(this));
    this.setHandler('cancel', this.onCancel.bind(this));
    this.activate();
    this.select(Math.max(this._data.length-1, 0));
    this.openness = 255;
  };
  Window_HS_BacklogList.prototype.maxItems = function(){ return this._data.length; };
  Window_HS_BacklogList.prototype.item     = function(i){ return this._data[i]; };

  // draw with control codes + shrink
  Window_HS_BacklogList.prototype.drawTextExClampedFS = function(text, x, y, width, maxHeight, baseFs){
    const rendered = this.convertEscapeCharacters(String(text||''));
    this.resetFontSettings();
    let fs = baseFs; this.contents.fontSize = fs;
    if (SHRINK){
      let ts = this.createTextState(rendered, x, y, width);
      let h  = this.calcTextHeight(ts, true);
      let n=0;
      while (h > maxHeight && fs > 12 && n < 12){
        fs -= 2; n++;
        this.resetFontSettings();
        this.contents.fontSize = fs;
        ts = this.createTextState(rendered, x, y, width);
        h  = this.calcTextHeight(ts, true);
      }
    }
    this.drawTextEx(rendered, x, y, width);
  };

  Window_HS_BacklogList.prototype.drawItem = function(i){
    const e = this.item(i); if (!e) return;
    const r = this.itemLineRect(i);
    const left = r.x + 10;
    const top  = r.y + PADY + TEXT_OFSY;
    const w    = r.width - BULLET_RPAD;

    let y = top;

    // name
    if (SHOW_NAME && e.speaker){
      if (SEPARATE_NAME){
        this.changeTextColor(ColorManager.systemColor());
        const nameText = `【${e.speaker}】`;
        const nameMaxH = NAME_FS + 6;
        this.drawTextExClampedFS(nameText, left, y, w, nameMaxH, NAME_FS);
        this.resetTextColor();
        y += nameMaxH;
      } else {
        e._hsbl_head = `【${e.speaker}】 `;
      }
    }

    // body
    const body = (e._hsbl_head || '') + e.text;
    const maxH = (r.y + this.itemHeight()) - y - PADY - TEXT_OFSY;
    this.drawTextExClampedFS(body, left, y, w, Math.max(1, maxH), TEXT_FS_BASE);
    e._hsbl_head = null;

    // bullet
    if (e.voice){
      this.changeTextColor(ColorManager.systemColor());
      this.drawText('●', r.x + r.width - this.textWidth('●') - 8, r.y + 4, 24, 'right');
      this.resetTextColor();
    }
  };

  Window_HS_BacklogList.prototype.onOk = function(){
    const e = this.item(this.index());
    if (e && e.voice && window.HSBL && typeof window.HSBL.playFromEntry === 'function'){
      window.HSBL.playFromEntry(e);
    } else {
      SoundManager.playOk();
    }
    this.activate(); // keep selectable
  };
  Window_HS_BacklogList.prototype.onCancel = function(){ Overlay.requestClose(); };

  // ===== Overlay =====
  const Overlay = (function(){
    let bg=null, list=null, open=false, pendingClose=false;

    function safeAddChild(n){ if (n && SceneManager._scene?.addChild) SceneManager._scene.addChild(n); }
    function safeAddWindow(w){ if (!w) return; if (SceneManager._scene?.addWindow) SceneManager._scene.addWindow(w); else if (SceneManager._scene?.addChild) SceneManager._scene.addChild(w); }
    function safeRemove(n){
      if (!n) return;
      try{
        if ('deactivate' in n) try{ n.deactivate(); }catch(_){}
        n.visible=false;
        const p=n.parent;
        if (p?.removeChild) p.removeChild(n);
        else if (SceneManager._scene?._windowLayer) SceneManager._scene._windowLayer.removeChild(n);
        else if (SceneManager._scene?.removeChild) SceneManager._scene.removeChild(n);
      }catch(e){ console.warn('[HSBL-Core] remove failed:', e); }
    }

    function isOpen(){ return open; }
    function requestClose(){ pendingClose=true; }
    function _flush(){ if (pendingClose){ pendingClose=false; closeNow(); } }

    function openNow(){
      if (isOptionsOpenNow()) return;
      if (open) return;
      open=true; HSBL._openFlag=true;

      bg = new Sprite(new Bitmap(Graphics.width, Graphics.height));
      bg.bitmap.fillRect(0,0,Graphics.width,Graphics.height, ColorManager.dimColor1());
      bg.opacity = DIM_OPACITY;
      safeAddChild(bg);

      const ww=Graphics.boxWidth, wh=Graphics.boxHeight;
      const y = 24 + OVERLAY_OY;
      let   h = wh - 48 - OVERLAY_OY*2; if (h<100) h = Math.max(100, wh-48);
      const r = new Rectangle(24, y, ww-48, h);

      list = new Window_HS_BacklogList(r);
      safeAddWindow(list);
      list.activate();

      Input.clear?.(); TouchInput.clear?.();
    }
    function closeNow(){
      if (!open) return;
      open=false; HSBL._openFlag=false;
      safeRemove(list); list=null;
      safeRemove(bg);   bg=null;
      Input.clear?.(); TouchInput.clear?.();
    }

    return { open:openNow, requestClose, isOpen, _flush };
  })();

  // ===== Commands =====
  PluginManager.registerCommand(PN, 'OpenBacklog',  ()=> { if (allowOpen() && !isOptionsOpenNow()) Overlay.open(); });
  PluginManager.registerCommand(PN, 'CloseBacklog', ()=> Overlay.requestClose());
  PluginManager.registerCommand(PN, 'ClearBacklog', ()=> $gameSystem.hsClearBacklog());

  // ===== Input loop (right click safe-close) =====
  if (HOTKEY !== 'none'){
    const _Input_update = Input.update;
    Input.update = function(){
      _Input_update.apply(this, arguments);
      try{
        if (!Overlay.isOpen() && allowOpen() && !isOptionsOpenNow() && isHotkeySceneAllowed() && Input.isTriggered(HOTKEY)){
          Overlay.open(); Input.clear?.(); TouchInput.clear?.();
        }
        if (Overlay.isOpen() && (TouchInput.isCancelled() || Input.isTriggered('cancel'))){
          Overlay.requestClose(); Input.clear?.(); TouchInput.clear?.();
        }
      }catch(_){}
    };
  } else {
    const _Input_update2 = Input.update;
    Input.update = function(){
      _Input_update2.apply(this, arguments);
      try{
        if (Overlay.isOpen() && (TouchInput.isCancelled() || Input.isTriggered('cancel'))){
          Overlay.requestClose(); Input.clear?.(); TouchInput.clear?.();
        }
      }catch(_){}
    };
  }

  // ===== Switch watcher (NEW) =====
  let _prevOpenSw = null, _prevCloseSw = null;
  function watchSwitches(){
    if (!$gameSwitches) return;

    if (OPEN_SW > 0){
      const cur = !!$gameSwitches.value(OPEN_SW);
      if (_prevOpenSw === null) _prevOpenSw = cur;           // 初期化
      if (!Overlay.isOpen() && cur && !_prevOpenSw){         // 立ち上がりエッジ
        if (allowOpen() && !isOptionsOpenNow()) Overlay.open();
        if (OPEN_AUTO_OFF) $gameSwitches.setValue(OPEN_SW, false);
        Input.clear?.(); TouchInput.clear?.();
      }
      _prevOpenSw = cur;
    }

    if (CLOSE_SW > 0){
      const cur = !!$gameSwitches.value(CLOSE_SW);
      if (_prevCloseSw === null) _prevCloseSw = cur;
      if (Overlay.isOpen() && cur && !_prevCloseSw){         // 立ち上がりエッジ
        Overlay.requestClose();
        if (CLOSE_AUTO_OFF) $gameSwitches.setValue(CLOSE_SW, false);
        Input.clear?.(); TouchInput.clear?.();
      }
      _prevCloseSw = cur;
    }
  }

  // フレーム末：安全クローズ & スイッチ監視
  const _Scene_Base_update = Scene_Base.prototype.update;
  Scene_Base.prototype.update = function(){
    _Scene_Base_update.apply(this, arguments);
    try{ Overlay._flush(); }catch(_){}
    try{ watchSwitches(); }catch(_){}
  };

  // ===== Lockers（表示中のイベント停止） =====
  if (LOCK_INT){
    const _GI_update = Game_Interpreter.prototype.update;
    Game_Interpreter.prototype.update = function(){
      if (HSBL.isOpen()) return;
      _GI_update.apply(this, arguments);
    };
    const _GCE_update = Game_CommonEvent.prototype.update;
    Game_CommonEvent.prototype.update = function(){
      if (HSBL.isOpen()) return;
      _GCE_update.apply(this, arguments);
    };
  }
  if (LOCK_EVENTS){
    const _GE_update = Game_Event.prototype.update;
    Game_Event.prototype.update = function(){
      if (HSBL.isOpen()) return;
      _GE_update.apply(this, arguments);
    };
  }
  if (LOCK_PLAYER){
    const _GP_update = Game_Player.prototype.update;
    Game_Player.prototype.update = function(sceneActive){
      if (HSBL.isOpen()) return;
      _GP_update.apply(this, arguments);
    };
  }

})();

